<?php
/* NOTICE OF LICENSE
 *
 * This source file is subject to a commercial license from SARL SMC
 * Use, copy, modification or distribution of this source file without written
 * license agreement from the SARL SMC is strictly forbidden.
 * In order to obtain a license, please contact us: olivier@common-services.com
 * ...........................................................................
 * INFORMATION SUR LA LICENCE D'UTILISATION
 *
 * L'utilisation de ce fichier source est soumise a une licence commerciale
 * concedee par la societe SMC
 * Toute utilisation, reproduction, modification ou distribution du present
 * fichier source sans contrat de licence ecrit de la part de la SARL SMC est
 * expressement interdite.
 * Pour obtenir une licence, veuillez contacter la SARL SMC a l'adresse: olivier@common-services.com
 * ...........................................................................
 * @package    Amazon Market Place
 * @copyright  Copyright (c) 2011-2013 S.A.R.L SMC (http://www.common-services.com)
 * @copyright  Copyright (c) 2011-2011 Olivier B.
 * @author     Erick Turcios
 * @license    Commercial license
 * Support by mail  :  contact@common-services.com
 * Support on forum :  delete
 * Skype : delete13_fr
 * Phone : +33.970465505
 */

/**
 * Description of AmazonValidator
 *
 * @author Erick Turcios
 */
class AmazonValidator {
    private $throwException = false;
    private $errors=array();//stores exceptions raised, when $throwException = false
    private $elementRestrictions = array( 'type',
                                        'mandatory',
                                        'limit',
                                        'maxDigits',
                                        'maxLength',
                                        'minLength',
                                        'pattern',
                                        'minValue',
                                        'maxValue'                                        
                                      );
    private $allowedValues = 'allowedValues';
    private $elementValue = 'value';
    private $elementAttr = 'attr';
    
    private $instance;
    //initializes data to determine validation origin
    public function __construct($object, $throwException=false) {
        if($throwException):
            $this->throwException = true;
        endif;        
        $obj = clone $object; 
        $this->instance = get_object_vars($obj);        
    }
    
    /**
     * Receives an Instance of a Class to valide according to mainStructure defined in constructor
     * @param type $object
     */
    public function validateInstance($object){
        $productAttributes = get_object_vars($object);
        
        foreach($this->instance as $element=>$baseValue):                    
                $this->validateBaseStructure($element, $baseValue, $productAttributes[$element], $element);
        endforeach;
        
    }
    
    /**
     * This function is created to be called recursively when validating.
     * @param type $baseElement
     * @param type $baseValue
     * @param type $instanceValue
     * @param type $origin
     * @return boolean
     */
    private function validateBaseStructure($baseElement, $baseValue, $instanceValue, $origin){
        $x=null;
        $y=null;
        
        //Any attribute is a stdClass
        if($baseValue instanceof stdClass):
            $x = get_object_vars($baseValue);
            $this->validateElement($baseValue, $instanceValue, $origin);
        endif;
        
        if(is_array($baseValue)):
            $x = $baseValue;
        endif;
        
    }
    
    private function validateElement($baseValue, $instanceValue, $origin){
        $conditions = get_object_vars($baseValue);        
        $mandatory=false;
        $hasChildren = false;
        foreach($conditions as $key=>$val):
            if($val instanceof stdClass && $key!='attr'):
                $hasChildren = true;
                break;
            endif;
        endforeach;
        
        //Verifies if this array contains structure data
        foreach($conditions as $key=>$val):
            if($key=='mandatory'&&$val=='true'):
                $mandatory=true;
            elseif($key=='mandatory'&&$val=='false'):
                $mandatory=false;
            endif;
            $value=$this->getValue($instanceValue);
            if(in_array($key, $this->elementRestrictions)):                
                $this->validateRestriction($key,$val, $value, $this->getSequence($origin, $key), $mandatory, $hasChildren);
            endif;
        endforeach;  
        
        foreach($conditions as $key=>$val):
            $instance=null;
        
            if($key=='mandatory'&&$val=='true'):
                $mandatory=true;
            elseif($key=='mandatory'&&$val=='false'):
                $mandatory=false;
            endif;
            
            //Set value from instancex class
            if(is_array($instanceValue)):
                $arr = $instanceValue;
                if(isset($arr[$key])):
                    $instance = $arr[$key];                
                else:
                    $instance = null;                
                endif;
            elseif($instanceValue instanceof stdClass):
                $arr = get_object_vars($instanceValue);
                if(isset($arr[$key])):
                    $instance = $arr[$key];                
                else:
                    $instance = null;                
                endif;
            else:
                $instance=$instanceValue;
            endif;
            
            if(is_array($val)):
                $this->validateBaseStructure($key, $val, $instance, $this->getSequence($origin, $key));
            endif;
            if($val instanceof stdClass):
                //if($key=='attr'&&$this->getValue($instanceValue)!=null):
                    $this->validateElement($val, $instance, $this->getSequence($origin, $key));
                //endif;
            endif;    
        endforeach;
    }
     
    private function validateRestriction($key,$val, $instanceValue,  $origin, $parentMandatory, $hasChildren){
        
        switch($key):
            case 'mandatory': 
                if($val=='true'&&$instanceValue==null):
                    $this->raiseException ('Mandatory value must be entered', $origin);
                endif;
                break;
            case 'type': $this->validateType($val, $instanceValue, $origin); 
                break;
            case 'limit': //pending; 
                break;
            case 'maxDigits':
                         if($val==null):
                             break;
                         endif;
                         $p = strpos($instanceValue, '.'); //decimal position
                         if($p >0 &&strlen(substr($instanceValue, $p+1))>$val):
                             $this->raiseException ('Wrong number of digits.',$origin);
                         endif;
                         break;
            case 'maxLength':
                        if($val!=null&&strlen($instanceValue)>$val):
                             $this->raiseException ('Max Length is: '.$val, $origin);
                         endif;
                         break;
            case 'minLength':
                        if($val!=null&&strlen($instanceValue)<$val):
                             $this->raiseException ('Min Length is: '.$val, $origin);
                         endif;
                         break;
            case 'pattern':
                       if($val!=null&&preg_match('\''.trim($val).'\'', $instanceValue)==0):
                            $this->raiseException ('Not a valid pattern ('.$val.') for this field. ', $origin);
                       endif;
                         break;
            case 'minValue':
                         if($val!=null&&$instanceValue<$val):
                             $this->raiseException ('Min Value is:'.$val, $origin);
                         endif;
                         break;
            case 'maxValue':
                         if($val!=null&&$instanceValue>$val):
                             $this->raiseException ('Max Length is :'.$val, $origin);
                         endif;
                         break;
            case 'allowedValues':
                         $arr=null;
                         if(!is_array($val)):
                             $arr=array($val);
                         else:
                             $arr=$val;
                         endif;
                         if(empty($arr)):
                            break; 
                         endif;
                         
                         if($arr!=null&&  !in_array($instanceValue, $arr)):
                             $str=  implode(',', $arr);
                             $this->raiseException ($instanceValue.' is not any of permitted values: '.$str, $origin);
                         endif;
                         break;
            default: null;
        endswitch;
    }
    
    private function validateType($type, &$value, $origin){
        $t  = $type;
        $textPattern = '/[\0x\0x00\0x01\0x02\0x03\0x04\0x05\0x06\0x07\0x08\0x0B\0x0C\0x0E\0x0F\0x10\0x11\0x12\0x13\0x14\0x15\0x16\0x17\0x18\0x19\0x1A\0x1B\0x1C\0x1D\0x1E\0x1F\0x7F]+/';
        $numberPattern = '/[^0-9\.]+/';
        $date = array('date','time','dateTime','duration','gDay','gMonth',
                      'gMonthDay','gYear','gYearMonth');
        if(in_array($t,$date)):
            $t='date';
        endif;
        
        switch ($t):
            case 'text': preg_replace($textPattern, '', $value); break;
            case 'number': 
                    if(preg_match($numberPattern, $value)>0):
                         $this->raiseException('Number format Not Valid: '.$value, $origin);
                    endif;
                    break;
            case 'date': $this->validateDate($type, $value, $origin);
                       break;
            case 'boolean':
                    if(strtolower($value)!='true'&&strtolower($value)!='false'):
                        $this->raiseException('Value must TRUE or FALSE, found: '.$value, $origin);
                    endif;
                    break;
            default: 
                     $this->raiseException ('Type "'.$type.'" is not defined in this validation.', $origin);
        endswitch;
    }
    
    private function validateDate($type, $value, $origin){
        switch($type):
            case 'date':
                if(preg_match('/[0-9]{4}-(0[1-9]|1[0-2])-(0[1-9]|[1-2][0-9]|3[0-1])$/',$value)==0):
                    $this->raiseException ('Date format Not Valid: '.$value, $origin);
                endif;
                break;
            case 'time':
                if(preg_match('/[0-2]{2}:[0-9]{2}:[0-9]{2}$/',$value)==0):
                    $this->raiseException ('Date format Not Valid: '.$value, $origin);
                endif;
                break;
            case 'dateTime' : 
                if(preg_match('/[0-9]{4}-(0[1-9]|1[0-2])-(0[1-9]|[1-2][0-9]|3[0-1] [0-2]{2}:[0-9]{2}:[0-9]{2})$/',$value)==0):
                    $this->raiseException ('Date Time format Not Valid: '.$value, $origin);
                endif;
                break;
            case 'duration' :break; //need the regex to be included here
            case 'gDay' :
                if(preg_match('(\d{2})',$value)==0):
                    $this->raiseException ('Day format Not Valid: '.$value, $origin);
                endif;
                break;
            case 'gMonth' :
                if(preg_match('(\d{2})',$value)==0):
                    $this->raiseException ('Month format Not Valid: '.$value, $origin);
                endif;
                break;
            case 'gMonthDay' :
                if(preg_match('(\d{2})-(\d{2})',$value)==0):
                    $this->raiseException ('Month-Day format Not Valid: '.$value, $origin);
                endif;
                break;
            case 'gYear' :
                if(preg_match('(\d{4})',$value)==0):
                    $this->raiseException ('Year format Not Valid: '.$value, $origin);
                endif;
                break;
            case 'gYearMonth':
                if(preg_match('(\d{4})-(\d{2})',$value)==0):
                    $this->raiseException ('Year-Month format Not Valid: '.$value, $origin);
                endif;
                break;
            default: $this->raiseException ('Unknown Date format: '.$value, $origin);
        endswitch;
    }
    
    private function getSequence($previousCall, $currentCall){
        if(in_array($currentCall, $this->elementRestrictions)||
                $currentCall==$this->elementAttr):
            return $previousCall;
        endif;
        return $previousCall.'->'.$currentCall;
    }
    
    private function raiseException($error, $sequence){
        if($this->throwException):
            throw new Exception ($error.' Reference: '.$sequence);
        else:
            $arr = array($error, $sequence);
            if(!in_array($arr, $this->errors)):
                $this->errors[]=$arr;
            endif;
            
        endif;
    }
    private function getValue($instanceValue){        
        $value=null;
        if(isset($instanceValue->value)):
            $value=$instanceValue->value;
        else:
           if(!is_array($instanceValue)&&!($instanceValue instanceof stdClass)):
               $value=$instanceValue;
           endif; 
        endif;
        if($instanceValue instanceof stdClass):
            $x = get_object_vars($instanceValue);
            foreach($x as $child):
                if($child instanceof stdClass):
                    $value='Has children atributes'; //just to indicate it has a value, this is not displayed
                endif;
            endforeach;
        endif;
        return $value;
    }
    public function getErrors(){
        if(!$this->throwException):
            return $this->errors;
        endif;
        
        return null;
    }
        
}

?>
