<?php
// $Id: filesystem.inc.php 5 2008-12-12 05:55:19Z adgar $

/**
 * This class wrapper for standart filesystem function.
 * Originaly write by elf <zuenko@gmail.com>
 * 
 * Edited by Leonov Sergey <adgar@softservice.org> 
 * 28.11.2006
 * 
 * @version 1.0
 * @copyright Copyright (c) 2005-2006  by Zuenko E.
 * @access public
 * @package filesystem class
 */
class fs{
    
    /**
     * [filecreate]
     * Create file.
     * @example fs::filecreate("/simpledir" , "simplename.txt" , 777);
     * @param string  path
     * @param string  file name
     * @param int     chmod
     * @return operation info
     */
    function filecreate($path , $name , $chmod = false){
        $path = $path . "/" . basename($name);
        if (!file_exists($path)) {
            if ( ( $stream = fs::fsopen($path , "w") ) !== false) {
                fs::fsclose($stream);
                if ($chmod != false) {
                    if (@chmod($path , octdec($chmod))){
                        return true;
                    }
                }
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }
    /**
     * [dircreate]
     * Create dir.
     * @example fs::dircreate("/simpledir" , "simpledir" , 777);
     * @param string  path
     * @param string  file name
     * @param int     chmod
     * @return operation info
     */
    function dircreate($path , $name , $chmod = false){
        $path = $path . "/" . basename($name);
        if ( !file_exists($path) ) {
            if ($chmod != false) {
                if ( @mkdir($path , octdec($chmod)) ){
                    return true;
                }
            } else {
                if (@mkdir($path , octdec(664))) {
                    return true;
                }
            }
        } else {
            return false;
        }
        return false;
    }
    /**
     * [open]
     * Open a file or a URL and return a file pointer.
     * @example fs::open("/simpledir/simplefile.txt" , "r");
     * @param string  file name
     * @param string  open type
     * @return resource or operation info
     */
    function open($filename,$mode){
        if ( ($stream = @fopen($filename , $mode)) !== false ) {
            return $stream;
        } else {
            return false;
        }
    }
    /**
     * [close]
     * Close an open file pointer.
     * @example fs::close($stream);
     * @param string  file name
     * @param string  open type
     * @return resource or operation info
     */
    function close($stream){
        if ( @fclose($stream) ) {
            return true;
        } else {
            return false;
        }
    }
    /**
     * [read]
     * Read file if $length = 0 read all file.
     * @example fs::read($stream , 1024);
     * @param resource file pointer
     * @param int  lenght
     * @return string file content
     */
    function read($stream , $length = 0){
        if ( is_int($length) ) {
            if ($length == 0) {
                $buffer = "";
                while (!feof($stream)) {
                    $buffer .= @fread($stream , 1024);
                }
                return $buffer;
            } else {
                return @fread($stream , $length);
            }
        } else {
            return false;
        }
    }
    /**
     * [write]
     * Write to file.
     * @example fs::write($stream , "abcd" , true);
     * @param resource file pointer
     * @param string content to write
     * @param boolean use lock
     * @return boolean operation info
     */
    function write($stream , $content , $use_lock = true){
        if ($use_lock == true) flock($stream , LOCK_EX);
        
        if ( @fwrite($stream , $content ) ) 
            $writed = true;
        else
            $writed = false;
            
        if ($use_lock == true) flock($stream , LOCK_UN);
        return $writed;
    }
    /**
     * [delete]
     * Delete dir or file.
     * @example fs::delete("/simpledir");
     * @param string path
     * @return boolean operation info
     */
    function delete($path){
        if ( is_file($path) ) {
            
            if ( is_writeable($path) ){         
                if ( @unlink($path) ) {
                    return true;
                } else {
                    return false;
                }
            } else {
                return false;
            }   
        } else {
            if ( ( $handle = @opendir($path) ) === false) return false;
            rewinddir($handle);
            while ( ( $file = readdir($handle) ) !== false ) {
                if( $file == ".." || $file == "." ) continue;
                if (fs::delete( $path . "/" . $file ) !== true) {
                    return false;
                }
            }
            closedir($handle);
            if ( is_writeable($path) ) {
                @rmdir($path);
            } else {
                return false;
            }
            return true;
        }
    }
    /**
     * [copy]
     * Copy file or dir to new place.
     * @example fs::copy("/sourcedir","/destdir"); // Result "/destdir/sourcedir"
     * @param string source file or dir
     * @param string destination dir
     * @return boolean operation info
     */
    function copy($source, $destination, $mode=0664) {
        $destination = rtrim($destination, '/\\') . DIRECTORY_SEPARATOR;                
        if ( ( $source == $destination . '/' . basename($source) ) || ( $source == $destination ) || !is_readable($source)) {
            return false;
        }        
        $path = $destination . basename($source);
        if (is_file($source)) {
            $res = @copy($source , $path);
            @chmod($path, $mode);
            return $res;
        } else {
            $source = rtrim($source, '/\\') . DIRECTORY_SEPARATOR;
            if(!is_dir($path)) {
                @mkdir($path);
                @chmod($path, $mode);                
            }
            if ($dh = opendir($source)) {
                while (($file = readdir($dh)) !== false) {
                    if (('.' != $file) && ('..' != $file))
                        fs::copy($source . $file, $path, $mode);
                }
                closedir($dh);
            }
            return true;
        }
    }
    /**
     * [dirsize]
     * Calculates directory size.
     * @example fs::dirsize("/sourcedir");
     * @param string source file or dir
     * @return uinteger directory size
     */
    function dirsize($path){
        if ( !is_dir( $path ) ) return 0;
        $size = 0;
        if ( ( $handle = @opendir( $path ) ) === false) return false;
        rewinddir( $handle );
        while ( ( $file = readdir( $handle ) ) !== false ){
            if( $file == ".." || $file == "." ) continue;
            if ( is_file( $path . "/" . $file ) ) {
                $size += filesize($path . "/" . $file);
            } else {
                $subsize  = sprintf( "%u" , fs::fsdirsize( $path . "/" . $file ) );
                $size  += $subsize;
            }
        }
        closedir( $handle );
        return $size;
    }
    /**
     * [dircount]
     * Calculates count files in directory.
     * @example fs::dircount("/sourcedir");
     * @param string source file or dir
     * @return integer count files
     */
    function dircount($path){
        if ( !is_dir( $path ) ) return 0;
        $count = 0;
        if ( ( $handle = @opendir( $path ) ) === false) return false;
        rewinddir( $handle );
        while ( ($file = readdir( $handle ) ) !== false ){
            if( $file == ".." || $file == "." ) continue;
            
            if ( is_file( $path . "/" . $file ) ) {
                $count ++;
            } else {
                $subcount  = sprintf( "%u" , fs::fsdircount( $path . "/" . $file ) );
                $count  += $subcount;
            }
            
        }
        closedir( $handle );
        return $count;
    }
    /**
     * [tree]
     * Makes array with tree [file|dir|all].
     * @example fs::tree("/sourcedir" , "all");
     * @param string path to directory
     * @param string type of tree
     * @return array array filled paths
     */
    function tree($path , $type = "file"){
        if ( !is_dir($path) ) return array();
        
        $buffer = array();
        
        if ( ( $handle = @opendir($path) ) === false) return false;
        rewinddir($handle);
        while ( ($file = readdir( $handle ) ) !== false ){
            if( $file == ".." || $file == "." ) continue;
            if ($type == "file") {
                if ( is_file($path . "/" . $file) ) {
                    $buffer[] = $path . "/" . $file;
                } else {
                    $subbuffer  = fs::tree($path . "/" . $file , $type);
                    $buffer = array_merge($subbuffer , $buffer);
                }
            }
            if ($type == "dir") {
                if ( is_dir($path . "/" . $file) ) {
                    $buffer[] = $path . "/" . $file;
                    $subbuffer  = fs::tree($path . "/" . $file , $type);
                    $buffer = array_merge($subbuffer , $buffer);
                }
            }
            if ($type == "all") {
                if (is_dir($path . "/" . $file)) {
                    $buffer[] = $path . "/" . $file;    
                    $subbuffer  = fs::tree($path . "/" . $file , $type);
                    $buffer = array_merge($subbuffer , $buffer);
                } else {
                    $buffer[] = $path . "/" . $file;
                }
            }
        }
        closedir( $handle );
        return $buffer;
    }
    /**
     * [dirget]
     * Scan directory and make array.
     * @example fs::dirget("/sourcedir");
     * @param string path to directory
     * @return array array filled paths
     */
    function dirget($path, $preg_pattern='') {
        if ( !is_dir($path) ) return array();
        
        $buffer = array();
        
        if ( ( $handle = @opendir($path) ) === false) return false;
        rewinddir($handle);
        while ( ($file = readdir( $handle ) ) !== false ) {
            if( $file == ".." || $file == "." ) continue;
            if (!empty($preg_pattern)) {
                if (preg_match($preg_pattern, $file)) $buffer[] =  $file;
            } else {
                $buffer[] =  $file;
            }
        }
        closedir( $handle );
        return $buffer;
    }
    /**
     * [info]
     * Scan directory and make array.
     * @example fs::info("/sourcedir" , true);
     * @param string path to directory
     * @param boolean calculate directory size?
     * @return array array with element info
     */
    function info($path , $dirsize = false){
        $info         = array();
        $info['name'] = basename( $path );
        $info['path'] = $path;
        if ( is_dir($path) ) {
            if ($dirsize == true) {
                $info['size'] = sprintf( "%u" , fs::fsdirsize($path) );
            } else {
                $info['size'] = 0;
            }
            $info['extension']   = "dir";
            $info['is_dir']      = true;
            $info['is_file']     = false;
        } else {
            eregi("\.([^.]+)$" , $path , $type);
            $info['size']        = sprintf( "%u" , filesize($path) );
            $info['extension']   = $type[1];
            $info['is_dir']      = false;
            $info['is_file']     = true;
        }
        
        $stat               = stat($path);
        $info['lastaccess'] = $stat[8];
        $info['lastmod']    = $stat[9];
        $info['chmod']      = substr( sprintf( '%o' , $stat[2] ) , -4 );
        return $info;
    }
    
}
/**
 * End Class
 */
?>